/*
 * Copyright (c) 2006, Wygwam
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * - Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation and/or 
 * other materials provided with the distribution.
 * - Neither the name of Wygwam nor the names of its contributors may be 
 * used to endorse or promote products derived from this software without 
 * specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.openxml4j.opc;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;

import org.openxml4j.exceptions.InvalidFormatException;
import org.openxml4j.exceptions.InvalidOperationException;
import org.openxml4j.exceptions.OpenXML4JException;
import org.openxml4j.opc.internal.marshallers.ZipPartMarshaller;

/**
 * Zip implementation of a PackagePart.
 * 
 * @author Julien Chable
 * @version 1.0
 * @see PackagePart
 */
public class ZipPackagePart extends PackagePart {

	/**
	 * The zip entry corresponding to this part.
	 */
	private ZipEntry zipEntry;

	/**
	 * Constructor.
	 * 
	 * @param container
	 *            The container package.
	 * @param partName
	 *            Part name.
	 * @param contentType
	 *            Content type.
	 * @throws InvalidFormatException
	 *             Throws if the content of this part invalid.
	 */
	public ZipPackagePart(Package container, PackagePartName partName,
			String contentType) throws InvalidFormatException {
		super(container, partName, contentType);
	}

	/**
	 * Constructor.
	 * 
	 * @param container
	 *            The container package.
	 * @param zipEntry
	 *            The zip entry corresponding to this part.
	 * @param partName
	 *            The part name.
	 * @param contentType
	 *            Content type.
	 * @throws InvalidFormatException
	 *             Throws if the content of this part is invalid.
	 */
	public ZipPackagePart(Package container, ZipEntry zipEntry,
			PackagePartName partName, String contentType)
			throws InvalidFormatException {
		super(container, partName, contentType);
		this.zipEntry = zipEntry;
	}

	/**
	 * Get the zip entry of this part.
	 * 
	 * @return The zip entry in the zip structure coresponding to this part.
	 */
	public ZipEntry getZipArchive() {
		return zipEntry;
	}

	/**
	 * Implementation of the getInputStream() which return the inputStream of
	 * this part zip entry.
	 * 
	 * @return Input stream of this part zip entry.
	 */
	@Override
	protected InputStream getInputStreamImpl() throws IOException {
		// We use the getInputStream() method from java.util.zip.ZipFile
		// class which return an InputStream to this part zip entry.
		return ((ZipPackage) container).getZipArchive()
				.getInputStream(zipEntry);
	}

	/**
	 * Implementation of the getOutputStream(). Return <b>null</b>. Normally
	 * will never be called since the MemoryPackage is use instead.
	 * 
	 * @return <b>null</b>
	 */
	@Override
	protected OutputStream getOutputStreamImpl() {
		return null;
	}

	@Override
	public boolean save(OutputStream os) throws OpenXML4JException {
		return new ZipPartMarshaller().marshall(this, os);
	}

	@Override
	public boolean load(InputStream ios) throws InvalidFormatException {
		throw new InvalidOperationException("Method not implemented !");
	}

	@Override
	public void close() {
		throw new InvalidOperationException("Method not implemented !");
	}

	@Override
	public void flush() {
		throw new InvalidOperationException("Method not implemented !");
	}
}
